unit Umidi;
{
Zarlino - application permettant d'exprimenter diverses  solutions
dans le domaine de l'intonation juste.
Copyright (C) 1998 Y. Ouvrard, J. P. Vidal, O. Bettens.
Ce programme est un logiciel libre ; vous pouvez le redistribuer et/ou le
modifier conformment aux dispositions de la Licence Publique Gnrale GNU,
telle que publie par la Free Software Foundation ; version 2 de la licence,
ou encore ( votre choix) toute version ultrieure.
Ce programme est distribu dans l'espoir qu'il sera utile, mais SANS AUCUNE
GARANTIE ; sans mme la garantie implicite de COMMERCIALISATION ou D'ADAPTATION
A UN OBJET PARTICULIER.
Pour plus de dtail, voir la Licence Publique Gnrale GNU .
Vous devez avoir reu un exemplaire de la Licence Publique Gnrale GNU en mme
temps que ce programme ; si ce n'est pas le cas, crivez  la
Free Software Foundation Inc., 675 Mass Ave, Cambridge, MA 02139, Etats-Unis.
Pour tout contact avec les auteurs : zarlino@iname.com
}

interface

Uses
  WinTypes, WinProcs, Classes, Dialogs, MMSystem, Grids,

  UTempereNote, UEvtMidi;

Const
  VersionMidi = '1.24';
  NonLiee = False;
  Liee = True;

Type
  TNotes = (ut, utdiese, re, rediese, mi, fa, fadiese, sol, soldiese, la, ladiese, si);

  TControle = (Modulation, Volume, Balance, Expression, Reverberation, Chorus);

  TMidi = class
  { public }
    hMidi: Integer;
    Etat: Integer;
    Constructor Create;
    Function Free: Integer;
    Function Out(Msg: dWord): Integer;
    Function Init: Integer;
    Procedure NoteOff(Canal: Byte; Note: TNotes; Octave: ShortInt);
    Procedure NoteOn(Canal: Byte; Note: TNotes; Octave: ShortInt; Attaque: Byte; Correction: Real);
    Procedure Diapason(Canal: Byte; Correction: Real);
    Procedure ProgramChange(Canal, Instrument: Byte);
    Procedure ChannelPressureChange(Canal, Valeur: Byte);
    Procedure PitchBendChange(Canal, LSB, MSB: Byte);
    Procedure AllNoteOff(Canal: Byte);
    Procedure ControlChange(Canal: Byte; TypeControle: TControle; Valeur: Byte);
    (*
    Procedure SensibiliteDiapason(Canal, PlageDemiTons, Cents, DemITons: Byte);
    Procedure RAZSensibiliteDiapason(Canal: Byte);
    Procedure RPN(Canal, MSB, LSB: Byte);
    *)
    Procedure ModulationChange(Canal: Byte; Valeur: Byte);
    Procedure VolumeChange(Canal: Byte; Valeur: Byte);
    Procedure BalanceChange(Canal: Byte; Valeur: Byte);
    Procedure ExpressionChange(Canal: Byte; Valeur: Byte);
    Procedure ReverberationChange(Canal: Byte; Valeur: Byte);
    Procedure ChorusChange(Canal: Byte; Valeur: Byte);
  End;

Const
  NomNotes: Array[TNotes] of String[4] = ('ut', 'ut#', 'r', 'r#', 'mi', 'fa', 'fa#', 'sol', 'sol#', 'la', 'la#', 'si');
  NotesLettres: Array[TNotes] of String[2] = ('C', 'C#', 'D', 'D#', 'E', 'F', 'F#', 'G', 'G#', 'A', 'A#', 'B');

Type
  TPiste = Class(TMemoryStream)
    FCanal: Byte;
    Total: LongInt;
    MemoCorrection,
    MemoDuree: Real;
    lgDeltaTime, lgNoteOff: Byte;
    Buffer: Array[1..4] of Byte;
    DeltaTimeEcrit: Boolean;
    Constructor Create(Canal: Byte);
    Procedure Ecrire(const Buffer; Compte: Integer);
    Procedure Rembobine(Octets: Integer);
    Procedure EcrireNote(NumeroNote, Attaque: Byte);
    Procedure EcrireNoteOff(NumeroNote: Byte);
    Procedure EcrireInstrument(Instrument: Byte);
    Procedure EcrireDeltaTime(Duree: Real);
    Procedure EcrirePitch(Correction: Real);
  End;

  TEnregistreurMidi = Class
    NomFichier: ShortString;
    nbPistes: Byte;
    MsHeader: TMemoryStream;
    MsTimeTempo: TMemoryStream;
    Piste: Array[0..15] of TPiste;
    Constructor Create(nom_du_fichier: ShortString; numerateur_mesure, denominateur_mesure, tempo: Byte);
    Procedure Free;
    Procedure EnregistreNote
      (Canal: Byte; Note: TNotes; Octave: ShortInt; Attaque: Byte; Correction, Duree: Real; aLiee: Boolean);
    Procedure CreerPiste(Var Canal: Byte);
    Procedure EnregistreSilence(Canal: Byte; Duree: Real);
    Procedure EnregistreInstrument(Canal, Instrument: Byte);
    Procedure EnregistrerFichierMidi;
  End;

  TLecteurMidi = Class
    NomFichier: ShortString;
    nbPistes, numerateur_mesure, denominateur_mesure, tempo, armature, majeurmineur: Byte;
    Tms: TMemoryStream;
    Buffer: TBuffer;
    FinPiste: Boolean;
    FormatFichier, noPiste, UniteDelta: Integer;
    Constructor Create(nom_du_fichier: ShortString);
    Procedure Free;
    Procedure CreerListeNotes;
  End;

Var
  Midi: Tmidi;

implementation

Const
  MidiNoteOff = $80;
  MidiNoteOn = $90;
  MidiChannelModeMessage = $B0;
  MidiControlChange = $B0;
  MidiProgramChange = $C0;
  MidiChannelPressure = $D0;
  MidiPitchBendChange = $E0;
  MidirpnMSB = $65;
  MidirpnLSB = $64;

  UniteDeltaTime = 480;

  ModeleHeader: Array[1..14] of Char = ('M', 'T', 'h', 'd', #0, #0, #0, #6, #0, #1, #0, #0, #1, #$E0);
  ModeleTimeTempo: Array[1..27] of Char = ('M', 'T', 'r', 'k', #0, #0, #0, #$13,
    #0, #$FF, #$58, #$4, #0, #0, #$18, #8, #0, #$FF, #$51, #3, #0, #0, #0, #0, #$FF, #$2F, #0);
  ModelePisteNotes: Array[1..8] of Char = ('M', 'T', 'r', 'k', #0, #0, #0, #0);
  FinDePiste: Array[1..4] of Char = (#0, #$FF, #$2F, #$0);

  OffsetNbPistes = 10;
  OffsetTimeSignature = 12;
  OffsetTempo = 20;

{ ------------- TLecteurMidi --------------- }

Constructor TLecteurMidi.Create(nom_du_fichier: ShortString);
Begin
  inherited Create;
  NomFichier := nom_du_fichier;
  Tms := TMemoryStream.Create;
  Tms.LoadFromFile(NomFichier);
  CreerListeNotes;
End;

Procedure TLecteurMidi.Free;
Begin
  Tms.Clear;
  Tms.Free;
  ListeNote.Free;
  inherited Free;
End;

Procedure TLecteurMidi.CreerListeNotes;
Var
  Signature: String[4];
  Taille: LongInt;

  Function OctetSuivant: Byte;
  Begin
    Tms.Read(Result, 1);
    Dec(Taille);
  End;

  Procedure LireHeader(Taille: LongInt);
  Begin
    Tms.Read(Buffer, Taille);
    FormatFichier := 16 * Buffer[0] + Buffer[1];
    nbPistes := 16 * Buffer[2] + Buffer[3];
    If Buffer[4] and $80 = 0 Then
      UniteDelta := 256 * Buffer[4] + Buffer[5]
    Else
      UniteDelta := Buffer[4] * ((not(Buffer[5] and $7F) and $7F) + 1); {  revoir }
  End; { LireHeader }

  Procedure LirePiste(Taille: LongInt);
  Var
    Delta: LongInt;
    Octet, Canal: Byte;
    Tempo: LongInt;

  Procedure EditeMeta;
  Var
    nb: Integer;
    mM: Boolean;
    TypeMeta, Ton: Byte;

    Procedure EditeHex;
    Var
      Ix: Integer;
    Begin { EditeHex }
      nb := OctetSuivant;
      For Ix := 1 to nb do OctetSuivant;
    End; { EditeHex }

    Begin { EditeMeta }
      TypeMeta := OctetSuivant;
      Case TypeMeta of
        $00: Editehex;
        $01..$07: Begin { Texte }
          EditeHex;
        End;
        $20: EditeHex;
        $2F: FinPiste := OctetSuivant = 0;
        $51: Begin { Tempo }
          OctetSuivant;
          Tempo := OctetSuivant shl 8;
          Tempo := (Tempo + OctetSuivant) shl 8;
          Tempo := Tempo + OctetSuivant;
        End;
        $54: EditeHex;
        $58: Begin
          Octet := OctetSuivant;
          numerateur_mesure := OctetSuivant;
          denominateur_mesure := 1 shl OctetSuivant;
          Octet := OctetSuivant; { messages / battement }
          Octet := OctetSuivant; { triples/noire }
        End;
        $59: Begin
          OctetSuivant; { Tonalit/Gamme }
          Ton := OctetSuivant;
          mM := OctetSuivant = 1; { vrai si mineur }
        End;
      Else { Case }
        EditeHex;
      End; { Case }
    End; { EditeMeta }

    Procedure EditeNote(aNote: Byte);
    Var
      OnOff: Boolean;
	    Note, Velocite: Byte;
    Begin { EditeNote }
    	If (aNote and $80)  = $80 Then Begin
  	  	Canal := Succ(aNote and $0F);
	      Note := OctetSuivant;
      End Else
      	Note := aNote;
     	Velocite := OctetSuivant;
      OnOff := (Velocite > 0) and not ((aNote and $90) = $80);
  		ListeNote.AjouterNote(Delta, Canal, Note, Velocite, OnOff, True);
    End; { EditeNote }

    Procedure LireDeltaTime;
    Begin { LireDeltaTime }
      Delta := 0;
      Repeat
        Octet := OctetSuivant;
        Delta := Delta shl 7 + (Octet and $7F);
      Until Octet < 128;
    End; { LireDeltaTime }

    Procedure LireEvenement;
    Begin { LireEvenement }
      Octet := OctetSuivant;
      If Octet in [$A0..$DF] Then ListeNote.AjouterNote(Delta, Succ(Octet and $0F), 0, 0, True, False);
      Case Octet of
        $00..$9F: EditeNote(Octet);
        $A0..$AF, $B0..$BF, $E0..$EF: Begin OctetSuivant; OctetSuivant; End;
        $C0..$CF, $D0..$DF: OctetSuivant;
        $F0: Repeat Octet := OctetSuivant Until Octet = $F7;
        $FF: EditeMeta;
      End; { Case }
    End; { LireEvenement }

  Begin { LirePiste }
    FinPiste := False;
    Repeat
      LireDeltaTime;
      LireEvenement;
    Until FinPiste or (Taille = 0);
  End; { LirePiste }

  Procedure LireChunk;
  Var
    Ix: Integer;
  Begin { LireChunk }
    Tms.Read(Buffer, 4);
    For Ix := 1 to 4 do Signature[Ix] := Chr(Buffer[Ix-1]);
    Tms.Read(Buffer, 4);
    Taille := 65536 * Buffer[0] + 4096 * Buffer[1] + 256 * Buffer[2] + Buffer[3];
    If Signature = 'MThd' Then
      LireHeader(Taille)
    Else
      If Signature = 'MTrk' Then Begin
        Inc(noPiste);
        LirePiste(Taille);
      End Else Begin
        Tms.Read(Buffer, Taille);
      End;
  End; { LireChunk }

Begin { CreerListeNotes }
  Signature := '1234';
  Tms.Seek(0, 0);
  noPiste := 0;
  ListeNote := TListeNote.Create;
  Repeat
    LireChunk;
  Until (Tms.Position >= Tms.Size);
  ListeNote.Trier;
End; { CreerListeNotes }

{ ------------- TPiste --------------- }

Constructor TPiste.Create(Canal: Byte);
Begin
  inherited Create;
  FCanal := Canal;
  Write(ModelePisteNotes, SizeOf(ModelePisteNotes));
  Total := 0;
  MemoCorrection := 0;
  MemoDuree := 0;
  lgDeltaTime := 0;
  lgNoteOff := 0;
  DeltaTimeEcrit := False;
End;

Procedure TPiste.Ecrire(const Buffer; Compte: Integer);
Begin
  Write(Buffer, Compte);
  Inc(Total, Compte);
End;

Procedure TPiste.Rembobine(Octets: Integer);
Begin
  Seek(-Octets, soFromEnd);
  Dec(Total, Octets);
End;

Procedure TPiste.EcrireNote(NumeroNote, Attaque: Byte);
Begin
  If not DeltaTimeEcrit Then EcrireDeltaTime(0);
  Buffer[1] := MidiNoteOn or FCanal;
  Buffer[2] := NumeroNote;
  Buffer[3] := Pred(Attaque);
  Ecrire(Buffer, 3); { 9x nn vv }
  DeltaTimeEcrit := False;
End;

Procedure TPiste.EcrireNoteOff(NumeroNote: Byte);
Begin
  Buffer[1] := MidiNoteOn or FCanal;
  Buffer[2] := NumeroNote;
  Buffer[3] := 0; { Note on avec vlocit 0 }
  Ecrire(Buffer, 3);
  lgNoteOff := 3;
  DeltaTimeEcrit := False;
End;

Procedure TPiste.EcrireInstrument(Instrument: Byte);
Begin
  Buffer[1] := MidiProgramChange or FCanal;
  Buffer[2] := Pred(Instrument);
  Ecrire(Buffer, 2);
  DeltaTimeEcrit := False;
End;

Procedure TPiste.EcrireDeltaTime(Duree: Real);
Var
  DeltaTime: Integer;
  n: Integer;
Begin
  MemoDuree := MemoDuree + Duree;
  DeltaTime := Round(MemoDuree * UniteDeltaTime);
  n := 4;
  Repeat
    Buffer[n] := (DeltaTime and $7F) or $80;
    DeltaTime := DeltaTime shr 7;
    Dec(n);
  Until DeltaTime = 0;
  Buffer[4] := Buffer[4] and $7F;
  lgDeltaTime := 4 - n;
  Ecrire(Buffer[Succ(n)], lgDeltaTime); { criture delta time }
  DeltaTimeEcrit := True;
End;

Procedure TPiste.EcrirePitch(Correction: Real);
Begin
  If not DeltaTimeEcrit Then EcrireDeltaTime(0);
  MemoCorrection := Correction;
  Correction := Correction * Etendue_Pitch_Unites / Etendue_Pitch_Cents + Etendue_Pitch_Unites div 2;
  Buffer[1] := MidiPitchBendChange or FCanal;
  Buffer[2] := Round(Correction) mod 128;
  Buffer[3] := Round(Correction) div 128;
  Ecrire(Buffer, 3); { Ex lsb msb }
  DeltaTimeEcrit := False;
End;

{ ---------- TEnregistreurMidi --------- }

Constructor TEnregistreurMidi.Create(nom_du_fichier: ShortString; numerateur_mesure, denominateur_mesure, tempo: Byte);
Var
  l_tempo, n: Integer;
Begin
  inherited Create;
  NomFichier := nom_du_fichier;
  nbPistes := 1; { au moins la piste time/tempo }
  MsHeader := TMemoryStream.Create;
  MsHeader.Write(ModeleHeader, SizeOf(ModeleHeader));
  MsTimeTempo := TMemoryStream.Create;
  With MsTimeTempo do Begin
    Write(ModeleTimeTempo, SizeOf(ModeleTimeTempo));
    Seek(OffsetTimeSignature, soFromBeginning);
    Write(numerateur_mesure, 1);
    n := 0;
    While denominateur_mesure > 1 do Begin
      Inc(n);
      denominateur_mesure := denominateur_mesure Shr 1;
    End;
    Write(n, 1);
    Seek(OffsetTempo, soFromBeginning);
    l_tempo := Round(60000000 / tempo) ; { noires / min --> s par noire }
    n := l_Tempo shr 16;
    Write(n, 1);
    n := (l_Tempo shr 8) and $FF;
    Write(n, 1);
    n := l_Tempo and $FF;
    Write(n, 1);
  End;
End;

Procedure TEnregistreurMidi.Free;
Var
  Ix: Integer;
Begin
  For Ix := 15 DownTo 0 do Piste[Ix].Free;
  MsTimeTempo.Free;
  MsHeader.Free;
  inherited Free;
End;

Procedure TEnregistreurMidi.CreerPiste(Var Canal: Byte);
Begin
  Canal := Pred(Canal);
  If Piste[Canal] = Nil Then Begin { crer nouvelle piste }
    Inc(nbPistes);
    Piste[Canal] := TPiste.Create(Canal);
  End;
End;

Procedure TEnregistreurMidi.EnregistreNote
  (Canal: Byte; Note: TNotes; Octave: ShortInt; Attaque: Byte; Correction, Duree: Real; aLiee: Boolean);
Var
  NumeroNote: LongInt;

Begin
  CreerPiste(Canal);

  NumeroNote := (Octave + 2) * 12 + ord(Note);

  { correction }
  With Piste[Canal] do Begin
    If aLiee Then Begin
      If Correction = MemoCorrection Then Begin { = }
        Rembobine(lgDeltaTime + lgNoteOff);
      End Else Begin { ~ }
        Rembobine(lgNoteOff);
        DeltaTimeEcrit := True;
        MemoDuree := 0; { il n'y a plus  revenir sur la note prcdente }
        EcrirePitch(Correction);
      End;
    End Else Begin { non lie }
      MemoDuree := 0;
      EcrirePitch(Correction);
      EcrireNote(NumeroNote, Attaque);
    End;
    EcrireDeltaTime(Duree);
    EcrireNoteOff(NumeroNote);
  End;
End;

Procedure TEnregistreurMidi.EnregistreSilence(Canal: Byte; Duree: Real);
Begin
  CreerPiste(Canal);

  With Piste[Canal] do Begin
    If DeltaTimeEcrit Then Begin
      Rembobine(lgDeltaTime);
      DeltaTimeEcrit := False;
    End Else
      MemoDuree := 0;
    EcrireDeltaTime(Duree);
  End;
End;

Procedure TEnregistreurMidi.EnregistreInstrument(Canal, Instrument: Byte);
Begin
  CreerPiste(Canal);

  With Piste[Canal] do Begin
    If not DeltaTimeEcrit Then Begin
      EcrireDeltaTime(0);
      DeltaTimeEcrit := True;
    End;
    EcrireInstrument(Instrument);
  End;
End;

Procedure TEnregistreurMidi.EnregistrerFichierMidi;
Var
  Tms: TMemoryStream;
  Canal: Integer;
  n: Byte;
Begin
  Tms := TMemoryStream.Create;
  With MsHeader do Begin
    Seek(OffsetNbPistes, soFromBeginning);
    n := nbPistes div 256;
    Write(n, 1);
    n := nbPistes mod 256;
    Write(n, 1);
    Tms.Write(Memory^, Size);
  End;
  With MsTimeTempo do Tms.Write(Memory^, Size);
  For Canal := 0 to 15 do If Piste[Canal] <> Nil Then with Piste[Canal] do Begin
    Seek(4, soFromBeginning);
    Inc(Total, 4); { la fin de piste que l'on crira directement sur Tms}
    n := (Total shr 24) and $FF;
    Write(n, 1);
    n := (Total shr 16) and $FF;
    Write(n, 1);
    n := (Total shr 8) and $FF;
    Write(n, 1);
    n := Total and $FF;
    Write(n, 1);
    Tms.Write(Memory^, Size);
    Tms.Write(FinDePiste, 4);
  End;
  Tms.SaveToFile(NomFichier);
  Tms.Free;
End;

{ ---------- TMidi --------- }

Constructor TMidi.Create;
Begin
  inherited Create;
  Etat := Init;
End;

Function TMidi.Free: Integer;
// Erreurs MidiOutClose
// 65 MIDIERR_STILLPLAYING	Buffers are still in the queue.
//  5 MMSYSERR_INVALHANDLE	The specified device handle is invalid.
//  7 MMSYSERR_NOMEM	The system is unable to load mapper string description.
Var
  Canal: Byte;
Begin
  For Canal := 1 to 16 do AllNoteOff(Canal);
  Result := midiOutClose(hMidi);
  (*
  If Result <> 0 Then Begin
    Case Result of
      MIDIERR_STILLPLAYING: S := 'Tout n''est pas encore jou';
      MMSYSERR_INVALHANDLE: S := 'Ticket invalide';
      MMSYSERR_NOMEM: S := 'Le systme ne peut charger la chane de description du mapper';
    End; { Case }
    ShowMessage('Erreur MidiOutClose : ' + S);
  End;
  *)
  inherited Free;
End;

Function Tmidi.init: Integer;
// Erreurs midiOutOpen
// 68 MIDIERR_NODEVICE	No MIDI port was found. This error occurs only when the mapper is opened.
//  4 MMSYSERR_ALLOCATED	The specified resource is already allocated.
//  2 MMSYSERR_BADDEVICEID	The specified device identifier is out of range.
// 11 MMSYSERR_INVALPARAM	The specified pointer or structure is invalid.
//  7 MMSYSERR_NOMEM	The system is unable to allocate or lock memory.
Var
  Canal: Byte;
begin
  If midiOutGetNumDevs = 0 Then Begin
    ShowMessage('Aucun instrument MIDI n''a t trouv');
    Result := -1;
    EXIT;
  End;
  Result := midiOutOpen(@hMidi, MIDI_MAPPER, 0, 0, 0);
  (*
  If Result <> 0 Then Begin
    Case Result of
      MIDIERR_NODEVICE: S := 'Port MIDI non trouv';
      MMSYSERR_ALLOCATED: S := 'Ressource actuellement utilise par un autre logiciel';
      MMSYSERR_BADDEVICEID: S := 'Mauvais identificateur de priphrique';
      MMSYSERR_INVALPARAM: S := 'Pointeur ou structure invalide';
      MMSYSERR_NOMEM: S := 'Problme d''allocation ou de verrouillage mmoire';
    End; { Case }
    ShowMessage('Erreur midiOutOpen : ' + S);
  End Else *)For Canal := 1 to 16 do Begin
    AllNoteOff(Canal);
    ProgramChange(Canal, 1);
  End;
end;

Function Tmidi.Out(Msg: dWord): Integer;
// Erreurs midiOutShortMsg
// 70 MIDIERR_BADOPENMODE	The application sent a message without a status byte to a stream handle.
// 67 MIDIERR_NOTREADY	The hardware is busy with other data.
// 5  MMSYSERR_INVALHANDLE	The specified device handle is invalid.
Begin
  Result := midiOutShortMsg(hMidi, Msg);
  (*
  If Result <> 0 Then Begin
    Case Result of
      MIDIERR_BADOPENMODE: S := 'Octet de status absent';
      MIDIERR_NOTREADY: S := 'Priphrique occup';
      MMSYSERR_INVALHANDLE: S := 'Ticket de priphrique invalide';
    End; { Case }
    ShowMessage('Erreur midiOutShortMsg : ' + S);
  End;
  *)
end;

Procedure TMidi.NoteOff(Canal: Byte; Note: TNotes; Octave: ShortInt);
{ Canal : 1  16 | Note : ut = 0 .. si = 11 | Octave : ut3 seruure = dbut de l'octave 3 }
Var
  NumeroNote: LongInt;
Begin
  NumeroNote := (Octave + 2) * 12 + ord(Note);
  Out(MidiNoteOff or Pred(Canal) or (NumeroNote shl 8));
End;

Procedure TMidi.NoteOn(Canal: Byte; Note: TNotes; Octave: ShortInt; Attaque: Byte; Correction: Real);
{ Attaque de 1 (inaudible)  128 (trs brillant) en passant par 64, valeur la plus courante }
{ Correction est exprime en cents, traduit en valeur MIDI }
Var
  NumeroNote: LongInt;
Begin
  NumeroNote := (Octave + 2) * 12 + ord(Note);
  Diapason(Canal, Correction);
  Out(MidiNoteOn or Pred(Canal) or (NumeroNote shl 8) or (Pred(Attaque) shl 16));
End;

Procedure TMidi.Diapason(Canal: Byte; Correction: Real);
Begin
  Correction := Correction * Etendue_Pitch_Unites / Etendue_Pitch_Cents + Etendue_Pitch_Unites div 2;
  PitchBendChange(Canal, Round(Correction) mod 128, Round(Correction) div 128);
End;

Procedure TMidi.ProgramChange(Canal, Instrument: Byte);
Begin
  Out(MidiProgramChange or Pred(Canal) or (Pred(Instrument) shl 8));
End;

Procedure TMidi.ChannelPressureChange(Canal, Valeur: Byte);
Begin
  Out(MidiChannelPressure or Pred(Canal) or (Valeur shl 8));
End;

Procedure TMidi.PitchBendChange(Canal, LSB, MSB: Byte);
Begin
  Out(MidiPitchBendChange or Pred(Canal) or (LSB shl 8) or (MSB shl 16));
End;

Procedure TMidi.AllNoteOff(Canal: Byte);
Begin
  Out(MidiChannelModeMessage + Pred(Canal) + $7B shl 8);
End;

Procedure TMidi.ControlChange(Canal: Byte; TypeControle: TControle; Valeur: Byte);
{ Valeur de 1  128 }
Var
  Code: Byte;
Begin
  Code := 0;
  Case TypeControle of
    Modulation: Code := $01;
    Volume: Code := $07;
    Balance: Code := $0A;
    Expression: Code := $0B;
    Reverberation:Code := $5B;
    Chorus: Code := $5D;
  End; { Case }
  If Code > 0 Then Out(MidiControlChange or Pred(Canal) or (Code shl 8) or (Pred(Valeur) shl 16));
End;

(*Procedure TMidi.SensibiliteDiapason(Canal, PlageDemiTons, Cents, DemITons: Byte);
Begin
  RPN(Canal, 0, 0);

End;

Procedure TMidi.RAZSensibiliteDiapason(Canal: Byte);
Begin
  RPN(Canal, $7F, $7F);
End;

Procedure TMidi.RPN(Canal, MSB, LSB: Byte);
Begin
  Out(MidiControlChange or Pred(Canal) or (MidirpnMSB shl 8) or (MSB shl 16));
  Out(MidirpnLSB or (LSB shl 8));
End;*)

Procedure TMidi.ModulationChange(Canal: Byte; Valeur: Byte);
{ Valeur de 1  128 }
Begin
  ControlChange(Canal, Modulation, Valeur);
End;

Procedure TMidi.VolumeChange(Canal: Byte; Valeur: Byte);
{ Valeur de 1  128 }
Begin
  ControlChange(Canal, Volume, Valeur);
End;

Procedure TMidi.BalanceChange(Canal: Byte; Valeur: Byte);
{ Valeur de 1  128 }
Begin
  ControlChange(Canal, Balance, Valeur);
End;

Procedure TMidi.ExpressionChange(Canal: Byte; Valeur: Byte);
{ Valeur de 1  128 }
Begin
  ControlChange(Canal, Expression, Valeur);
End;

Procedure TMidi.ReverberationChange(Canal: Byte; Valeur: Byte);
{ Valeur de 1  128 }
Begin
  ControlChange(Canal, Reverberation, Valeur);
End;

Procedure TMidi.ChorusChange(Canal: Byte; Valeur: Byte);
{ Valeur de 1  128 }
Begin
  ControlChange(Canal, Chorus, Valeur);
End;

initialization
  Midi := TMidi.Create;
finalization
  Midi.Free;
end.
